/** @file   polygon.h
 * @brief   Declaration of Polygon - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_POLYGON_H
#define H_WWW_POLYGON_H

#include <vector>
#include "vec2d.h"
#include "Color.h"


namespace WeWantWar {


/** @class  Polygon
 * @brief   Represents the 2d-polygons
 * @author  Tomi Lamminsaari
 *
 * Some GameObjects like <code> Car </code> are polygonized objects. The
 * <code> PolygonObject </code> - class uses this Polygon class.
 */
class Polygon
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** @struct Vtx
   * @brief   Each vertex is stored in an instance of this structure.
   * @author  Tomi Lamminsaari
   */
  struct Vtx {
    /** The coordinates of this vertex */
    eng2d::Vec2D  coords;
    /** The texture coordinate of this vertex. */
    eng2d::Vec2D  tcoords;
    /** Color of this vertex */
    eng2d::Color  c;
  };
  



  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor. Constructs new polygon with (-1,-1), (-1,1), (1,1), (1,-1)
   * as its vertices.
   */
  Polygon();
  
	/** Constructs a polygon with 3 vertices
 	 * @param     rV1               Vertex 1
   * @param     rV2               Vertex 2
   * @param     rV3               Vertex 3
   */
	Polygon( const Vtx& rV1, const Vtx& rV2,
           const Vtx& rV3 );

  /** Constructs a polygon with 4 vertices.
   * @param     rV1               Vertex 1
   * @param     rV2               Vertex 2
   * @param     rV3               Vertex 3
   * @param     rV4               Vertex 4
   */
  Polygon( const Vtx& rV1, const Vtx& rV2,
           const Vtx& rV3, const Vtx& rV4 );
           

	/** Destructor
   */
	virtual ~Polygon();

	/** Copy constructor.
   * @param     rO                Reference to another Polygon
   */
  Polygon( const Polygon& rO );

	/** Assignment operator
   * @param     rO                Reference to another Polygon
   * @return    Reference to us.
   */
  Polygon& operator = ( const Polygon& rO );



  ///
  /// Methods
  /// =======

  /** Sets the texture this polygon uses. The texture is stored as index
   * to the table containing all the texture bitmaps.
   * @param     textureIndex      Index of the texture you'd like to use
   *                              for this polygon. If == -1, the
   *                              texture mapping will be disabled.
   */
  void texture( int textureIndex );
  
  /** Sets the drawing mode information. Use the POLYTYPE - constants
   * defined inside the Allegro such as POLYTYPE_FLAT or POLYTYPE_ATEX
   * @param     ptype             The drawing mode how this polygon should be
   *                              rendered.
   */
  void polytype( int ptype );
  
  /** Sets new data for the vertex.
   * @param     vindex            Index of the vertex
   * @param     rV                New data for the vertex
   */
  void setVertex( int vindex, const Vtx& rV );



  ///
  /// Getter methods
  /// ==============
  
  /** Returns the index'th vertex.
   * @param     index             Index of the vertex.
   * @param     pV                Pointer to vertex where the data will be
   *                              copied.
   */
  void getVertex( int index, Vtx* pV ) const;
  
  /** Returns the rotated version of the index'th vertex and places its
   * data to Allegro's V3D_f-structure.
   * @param     index             Index of the vertex
   * @param     pV                Pointer to V3D_f structure where the data
   *                              will be copied
   * @param     a                 Rotation angle
   */
  void getRotatedVertex_ex( int index, V3D_f* pV, float a ) const;
  
  /** Returns the number of vertices this polygon has.
   * @return    Number of vertices this polygon has. Usually either 3 or
   *            4.
   */
  int vertexCount() const;

  /** Returns the index of the texture being used for this polygon.
   * @return    Index of the texture or -1 if no texture is in use.
   */
  int texture() const;

  /** Returns the drawingmode
   * @return    The drawing mode. This is one of the Allegro's polytype
   *            values such as POLYTYPE_FLAT or POLYTYPE_ATEX
   */
  int polytype() const;

protected:

  ///
  /// Members
  /// =======

  /** This vector holds the vertex coordinates. */
  std::vector< Vtx > m_originalVertices;
  
  /** Index of the bitmap that works as our texture */
  int   m_texture;
  
  /** The drawing mode */
  int   m_polytype;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: polygon.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:10+02  lamminsa
 * Initial revision
 *
 */
 
